// --------------------------------- lpsdrvr.h -------------------------------------------
//
//  Include file for Linux LabBrick LPS socket definitions
//
//  (c) 2020-2025 by Vaunix Technology Corporation, all rights reserved
//
//  NJB  Version 1.1 LPS Ethernet Driver Interface Definitions
//-----------------------------------------------------------------------------
#ifndef __LPSDRVR_H__
#define __LPSDRVR_H__

// ---------- Include headers ----------------

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif


// ----------- Global Equates ------------
#define MAXDEVICES 		64
#define MAX_MODELNAME 	32
#define PROFILE_MAX 	50					// Older non-hires phase shifters can store 50 profile elements in their eeprom
#define PROFILE_MAX_RAM 1000				// New FW V2.x based phase shifters have a 1000 element RAM profile buffer
#define PROFILE_MAX_HR 	25					// HiRes phase shifters only save 25 elements in their eeprom
#define CHANNEL_MAX 	64					// The largest device today has 64 channels with a full set of expansion modules
#define MAX_SWVERSION   7
#define MAX_NETBUFF     16
#define PROFILE_EMPTY 	0xFFFFFFFF 			// we use this as our marker for an empty entry in the profile

// ----------- Data Types -----------------
#define DEVID unsigned int

// ----------- Profile Control -----------
#define PROFILE_ONCE    1                	// play the profile once
#define PROFILE_REPEAT  2                	// play the profile repeatedly
#define PROFILE_OFF     0                	// stop the profile

// Status returns for commands
#define LVSTATUS int

#define STATUS_OK 				0
#define INVALID_DEVID			0x80000000	// MSB is set if the device ID is invalid
#define BAD_PARAMETER			0x80010000	// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO				0x80020000	// a failure in the I/O subsystem
#define DEVICE_NOT_READY		0x80030000	// device isn't open, no handle, etc.
#define FEATURE_NOT_SUPPORTED	0x80040000	// the selected Lab Brick does not support this function
#define DATA_UNAVAILABLE      	0x80050000  // the requested data item is unavailable
                                          	// usually due to a ramp or profile where reporting times exceed timeout limits
#define DEV_TABLE_FULL			0x80060000  // the device table is full
#define BAD_IP_ADDRESS			0x80070000	// the IP address is invalid (e.g. wrong format, etc)


// Status returns for DevStatus
#define INVALID_DEVID 		0x80000000 		// MSB is set if the device ID is invalid [global]
#define DEV_CONNECTED 		0x00000001 		// LSB is set if a device is connected [global]
#define DEV_OPENED 			0x00000002 		// set if the device is opened [global]
#define SWP_ACTIVE 			0x00000004 		// set if the device is sweeping
#define SWP_UP 				0x00000008 		// set if the device is sweeping up in frequency
#define SWP_REPEAT 			0x00000010 		// set if the device is in continuous sweep mode
#define SWP_BIDIRECTIONAL	0x00000020		// set if the device is in bi-directional ramp mode
#define PROFILE_ACTIVE		0x00000040		// set if a profile is playing

// Internal values in DevStatus
#define DEV_LOCKED   		0x00002000 		// set if we don't want read thread updates of the device parameters
#define DEV_RDTHREAD   		0x00004000 		// set when the read thread is running [global]
#define DEV_V2FEATURES		0x00008000		// set for devices with V2 feature sets [global]
#define DEV_HIRES			0x00010000		// set for HiRes devices [global]
#define DEV_BUFEMPTY  		0x00020000		// set when the device buffers are empty


void fnLPS_SetTestMode(bool testmode);
int fnLPS_AddLPSDevice(char* deviceIP);
int fnLPS_RemoveLPSDevice(char* deviceIP);
int fnLPS_GetNumDevices(void);
int fnLPS_GetDevInfo(DEVID* ActiveDevices);
int fnLPS_InitDevice(DEVID deviceID);
int fnLPS_CloseDevice(DEVID deviceID);
int fnLPS_CheckDeviceReady(DEVID deviceID);
int fnLPS_GetModelName(DEVID deviceID, char* Model_Name);
int fnLPS_GetSerialNumber(DEVID deviceID);
int fnLPS_GetLibVersion(void);
int fnLPS_GetDeviceStatus(DEVID deviceID);
LVSTATUS fnLPS_GetIPMode(DEVID deviceID, int* respdata);
LVSTATUS fnLPS_GetIPAddress(DEVID deviceID, char* respdata);
LVSTATUS fnLPS_GetNetmask(DEVID deviceID, char* respdata);
LVSTATUS fnLPS_GetGateway(DEVID deviceID, char* respdata);


LVSTATUS fnLPS_SetChannel(DEVID deviceID, int channel);
LVSTATUS fnLPS_SetWorkingFrequency(DEVID deviceID, int frequency);
LVSTATUS fnLPS_SetPhaseAngle(DEVID deviceID, int phase);
LVSTATUS fnLPS_SetPhaseAngleQ(DEVID deviceID, int phase, int channel);
LVSTATUS fnLPS_SetPhaseAngleMC(DEVID deviceID, int phase, unsigned long long chmask);

LVSTATUS fnLPS_SetRampStart(DEVID deviceID, int rampstart);
LVSTATUS fnLPS_SetRampEnd(DEVID deviceID, int rampstop);
LVSTATUS fnLPS_SetPhaseAngleStep(DEVID deviceID, int phasestep);
LVSTATUS fnLPS_SetPhaseAngleStepTwo(DEVID deviceID, int phasestep2);
LVSTATUS fnLPS_SetDwellTime(DEVID deviceID, int dwelltime);
LVSTATUS fnLPS_SetDwellTimeTwo(DEVID deviceID, int dwelltime2);
LVSTATUS fnLPS_SetIdleTime(DEVID deviceID, int idletime);
LVSTATUS fnLPS_SetHoldTime(DEVID deviceID, int holdtime);

LVSTATUS fnLPS_SetProfileElement(DEVID deviceID, int index, int phase);
LVSTATUS fnLPS_SetProfileCount(DEVID deviceID, int profilecount);
LVSTATUS fnLPS_SetProfileIdleTime(DEVID deviceID, int idletime);
LVSTATUS fnLPS_SetProfileDwellTime(DEVID deviceID, int dwelltime);
LVSTATUS fnLPS_StartProfile(DEVID deviceID, int mode);
LVSTATUS fnLPS_StartProfileMC(DEVID deviceID, int mode, int chmask, bool delayed);

LVSTATUS fnLPS_SetRampDirection(DEVID deviceID, bool up);
LVSTATUS fnLPS_SetRampMode(DEVID deviceID, bool mode);
LVSTATUS fnLPS_SetRampBidirectional(DEVID deviceID, bool bidir_enable);
LVSTATUS fnLPS_StartRamp(DEVID deviceID, bool go);
LVSTATUS fnLPS_StartRampMC(DEVID deviceID, int mode, int chmask, bool deferred);

LVSTATUS fnLPS_SaveSettings(DEVID deviceID);


int fnLPS_GetWorkingFrequency(DEVID deviceID);
int fnLPS_GetMinWorkingFrequency(DEVID deviceID);
int fnLPS_GetMaxWorkingFrequency(DEVID deviceID);

int fnLPS_GetPhaseAngle(DEVID deviceID);
int fnLPS_GetRampStart(DEVID deviceID);
int fnLPS_GetRampEnd(DEVID deviceID);
int fnLPS_GetDwellTime(DEVID deviceID);
int fnLPS_GetDwellTimeTwo(DEVID deviceID);
int fnLPS_GetIdleTime(DEVID deviceID);
int fnLPS_GetHoldTime(DEVID deviceID);

int fnLPS_GetPhaseAngleStep(DEVID deviceID);
int fnLPS_GetPhaseAngleStepTwo(DEVID deviceID);

int fnLPS_GetProfileElement(DEVID deviceID, int index);
int fnLPS_GetProfileCount(DEVID deviceID);
int fnLPS_GetProfileDwellTime(DEVID deviceID);
int fnLPS_GetProfileIdleTime(DEVID deviceID);
int fnLPS_GetProfileIndex(DEVID deviceID);

int fnLPS_GetMaxPhaseShift(DEVID deviceID);
int fnLPS_GetMinPhaseShift(DEVID deviceID);
int fnLPS_GetMinPhaseStep(DEVID deviceID);
int fnLPS_GetFeatures(DEVID deviceID);
int fnLPS_GetNumChannels(DEVID deviceID);
int fnLPS_GetProfileMaxLength(DEVID deviceID);

char* fnLPS_perror(LVSTATUS status);

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // __LPSDRVR_H__
